package com.ehking.demo.utils;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

public abstract class FastJsonUtils {
    static final Logger LOGGER = LoggerFactory.getLogger(FastJsonUtils.class);

    public static JSONObject convert(InputStream inputStream) {
        byte[] buffer = new byte[1024];
        int len = 0;
        try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
            while ((len = inputStream.read(buffer, 0, buffer.length)) != -1) {
                out.write(buffer, 0, len);
            }
            byte[] bytes = out.toByteArray();
            String jsonStr = new String(bytes, StandardCharsets.UTF_8);
            JSONObject jsonObject = JSONObject.parseObject(jsonStr);
            return jsonObject;
        } catch (IOException e) {
            LOGGER.info("", e);
        }
        return null;
    }

    /**
     * 对 JSON 对象中的数据按键名 a-z 排序，并递归处理嵌套的 JSON 数据，最终拼接键值
     *
     * @param json         需要处理的 JSON 对象
     * @param excludeField 排序排队的字段
     * @return hmacSourceResult 通过入参返回排序后的拼接字符串
     */

    public static void generateHmacSource(JSONObject json, String[] excludeField, StringBuilder hmacSourceResult) {
        // 如果 excludeField 为 null，则初始化为空数组
        if (excludeField == null) {
            excludeField = new String[0];
        }

        Set<String> keys = json.keySet();
        List<String> list = new ArrayList<String>(keys);

        Collections.sort(list);
        for (String key : list) {
            boolean exclude = false;
            for (String field : excludeField) {
                if (key.equals(field)) {
                    exclude = true;
                    break;
                }
            }
            if (exclude) {
                continue;
            }
            Object obj = json.get(key);
            if (obj == null) {
                LOGGER.warn("value is null for key:[{}],json:[{}]", key, json.toJSONString());
                continue;
            }
            if (obj instanceof JSONObject) {
                //如果值为对像时，递归调用
                generateHmacSource((JSONObject) obj, excludeField, hmacSourceResult);
            } else if (obj instanceof JSONArray) {
                // 如果值是 JSONArray，处理数组中的每个元素
                JSONArray array = (JSONArray) obj;
                for (Object element : array) {
                    if (element instanceof JSONObject) {
                        // 如果元素是 JSONObject，递归调用
                        generateHmacSource((JSONObject) element, excludeField, hmacSourceResult);
                    } else if (StringUtils.isNotBlank(element.toString())) {
                        // 如果是普通值，直接拼接到结果中
                        hmacSourceResult.append(element.toString()).append("#");
                    }
                }
            } else {
                if (StringUtils.isNotBlank(obj.toString())) {
                    hmacSourceResult.append(obj).append("#");
                }
            }
        }
    }

}
